/*
  **
  ** conf.c
  **
  ** I originaly wrote conf.c as part of the drpoxy package 
  ** thanks to Matthew Pratt and others for their additions. 
  **
  ** Copyright 1999 Jeroen Vreeken (pe1rxq@chello.nl)
  **
  ** This software is licensed under the terms of the GNU General 
  ** Public License (GPL). Please see the file COPYING for details.
  ** 
  **
*/

/**
 * How to add a config option :
 *
 *   1. think twice, there are settings enough
 *
 *   2. add a field to 'struct config' (conf.h) and 'struct config conf'
 *
 *   4. add a entry to the config_params array below, if your
 *      option should be configurable by the config file.
 */ 
#include "motion.h"
#include "video.h"
#include "conf.h"
#include "track.h"

int configcontinue=1;
int confignochild=0;

struct config conf = {
	DEF_WIDTH,
	DEF_HEIGHT,
	DEF_QUALITY,
	DEF_CHANGES,
	0,
	1,
	0,
	0, /* snapshot_overwrite */
	DEF_GAP,
	DEF_MAXMPEGTIME, /* max_mpeg_time */
	1, /* locate */
	IN_DEFAULT,
	0,
	DEF_MAXFRAMERATE,
	0,
	0,
	0,
	DEF_NOISELEVEL,
	0, /* min gap */
	0, /* lightswitch */
	0, /* jpg_cleanup */
	1, /* night compensate */
	0, /* adjust rate*/
	0, /* real motion */
	0, /* old layout */
	0, /* low_cpu */
	0, /* nochild (runtime not an option) */
	0, /* autobright */
	VIDEO_DEVICE,
	NULL,	/* video loopback pipe */
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,	/* mask file */
	NULL,	/* db */
	NULL,	/* host */
	NULL,	/* user */
	NULL,	/* password */
	NULL,	/* onsave command */
	NULL,	/* onmpeg command */
	NULL,	/* AXIS ip address */
	NULL,	/* motion video pipe */
	NULL	/* realconfig file */
};



static void copy_bool(char *, void *);
static void copy_int(char *, void *);
static void copy_string(char *, void *);
static void config_realconfig(char *str, void *val);
static void usage(void);

config_param config_params[] = {
	{
	"always_changes",
	&conf.always_diff,
	copy_bool
	},
	{
	"deamon",
	&deamon,
	copy_bool
	},
	{
	"snapshots",
	&alarmtime,
	copy_int
	},
	{
	"threshold",
	&conf.max_changes,
	copy_int
	},
	{
	"framerate",
	&conf.frame_limit,
	copy_int
	},
	{
	"locate",
	&conf.locate,
	copy_bool
	},
	{
	"output_motion",
	&conf.motion_img,
	copy_bool
	},
	{
	"output_normal",
	&conf.new_img,
	copy_bool
	},
	{
	"target_dir",
	&conf.filepath,
	copy_string
	},
	{
	"videodevice",
	&conf.device,
	copy_string
	},
	{
	"input",
	&conf.input,
	copy_int
	},
	{
	"norm",
	&conf.norm,
	copy_int
	},
	{
	"quality",
	&conf.quality,
	copy_int
	},
	{
	"quiet",
	&conf.quiet,
	copy_bool
	},
	{
	"ppm",
	&conf.ppm,
	copy_bool
	},
	{
	"width",
	&conf.width,
	copy_int
	},
	{
	"height",
	&conf.height,
	copy_int
	},
	{
	"execute",
	&conf.externcommand,
	copy_string
	},
	{
	"mail",
	&conf.mail_address,
	copy_string
	},
	{
	"sms",
	&conf.sms_nr,
	copy_string
	},
	{
	"snapshot_overwrite",
	&conf.snap_overwrite,
	copy_bool
	},
	{
	"gap",
	&conf.gap,
	copy_int
	},
	{
	"max_mpeg_time",
	&conf.maxmpegtime,
	copy_int	
	},
	{
	"minimum_gap",
	&conf.mingap,
	copy_int
	},
	{
	"mpeg_encode",
	&conf.mpeg,
	copy_bool
	},
	{
	"noise_level",
	&conf.noise,
	copy_int
	},
	{
	"mask_file",
	&conf.mask_file,
	copy_string
	},
	{
	"lightswitch",
	&conf.lightswitch,
	copy_bool
	},
	{
	"video_pipe",
	&conf.vidpipe,
	copy_string
	},
	{
	"jpg_cleanup",
	&conf.jpg_cleanup,
	copy_bool
	},
	{
	"onsave",
	&conf.onsavecommand,
	copy_string
	},
	{
	"onmpeg",
	&conf.onmpegcommand,
	copy_string
	},
	{
	"axis",
	&conf.axis,
	copy_string
	},
	{
	"motion_video_pipe",
	&conf.motionvidpipe,
	copy_string
	},
	{
	"night_compensate",
	&conf.nightcomp,
	copy_bool
	},
	{
	"adjust_rate",
	&conf.adjustrate,
	copy_bool
	},
	{
	"realmotion",
	&conf.realmotion,
	copy_bool
	},
	{
	"oldlayout",
	&conf.oldlayout,
	copy_bool
	},
	{
	"low_cpu",
	&conf.low_cpu,
	copy_bool
	},
	{
	"realconfig",
	&conf.realconfig,
	config_realconfig
	},
	{
	"auto_brightness",
	&conf.autobright,
	copy_bool
	},
#ifdef HAVE_MYSQL
	{
	"mysql_db",
	&conf.mysql_db,
	copy_string
	},
	{
	"mysql_host",
	&conf.mysql_host,
	copy_string
	},
	{
	"mysql_user",
	&conf.mysql_user,
	copy_string
	},
	{
	"mysql_password",
	&conf.mysql_password,
	copy_string
	},
#endif
	{
	"track_port",
	&track.port,
	copy_string	
	},
	{
	"track_motorx",
	&track.motorx,
	copy_int
	},
	{
	"track_maxx",
	&track.maxx,
	copy_int
	},
	{
	"track_iomojo_id",
	&track.iomojo_id,
	copy_int
	},
  { NULL, NULL , NULL }
};


static void conf_cmdline (void)
{
	int c;

        while ((c=getopt(conf.argc, conf.argv, "A:a:BCc:d:DE:F:f:G:g:hi:lL:M:mNn:O:P:pQq:r:S:s:t:V:"))!=EOF) 
                switch (c) {
			case 'A':
				conf.axis=optarg;
				break;                                                    
                        case 'a':                                               
                                sscanf (optarg, "%d", &alarmtime);              
                                break;
			case 'B':
				conf.mpeg=1;
				break;                                          
                        case 'C':                                               
                                conf.always_diff=1;                                  
                                break;                                          
                        case 'c':                                               
                                sscanf (optarg, "%d", &conf.max_changes);            
                                break;                                          
                        case 'd':                                               
                                conf.device=optarg;                                  
                                break;                                          
                        case 'E':                                               
                                conf.externcommand=optarg;                           
                                break;                                          
                        case 'D':                                               
                                deamon=1;                                       
                                break;
			case 'F':
				conf.mask_file=optarg;
				break;                                          
                        case 'f':                                               
                                sscanf (optarg, "%d", &conf.frame_limit);            
                                break;
			case 'G':
				sscanf (optarg, "%d", &conf.mingap);
				break;                                          
                        case 'g':                                               
                                sscanf (optarg, "%d", &conf.gap);                    
                                break;                                          
                        case 'i':                                               
                                sscanf (optarg, "%d", &conf.input);                  
                                break;                                          
                        case 'l':                                               
                                conf.locate=0;                                       
                                break;
			case 'L':
				sscanf (optarg, "%d", &conf.noise);
				break;                                          
                        case 'M':                                               
                                conf.mail_address=optarg;                            
                                break;                                          
                        case 'm':                                               
                                conf.motion_img=1;                                   
                                break;                                          
                        case 'N':                                               
                                conf.new_img=0;                                      
                                break;                                          
                        case 'n':                                               
                                if (!strcmp(optarg, "pal")) conf.norm=NORM_PAL;
				if (!strcmp(optarg, "pal-nc")) conf.norm=NORM_PAL_NC;
                                if (!strcmp(optarg, "ntsc")) conf.norm=NORM_NTSC;
                                if (!strcmp(optarg, "secam")) conf.norm=NORM_SECAM; 
                                break;
			case 'O':
				conf.onsavecommand=optarg;
				break;
			case 'P':
				conf.vidpipe=optarg;
				break;
			case 'p':
				conf.ppm=1;
				break;                                          
                        case 'S':                                               
                                conf.sms_nr=optarg;                                  
                                break;                                          
                        case 'q':                                               
                                sscanf (optarg, "%d", &conf.quality);
                                break;
			case 'r':
				/* Should already be handled */
				break;
		        case 'Q':
			        conf.quiet=1;
			        break;
                        case 's':                                               
                                sscanf (optarg, "%dx%d", &conf.width, &conf.height);      
                                break;                                          
                        case 't':                                               
                                conf.filepath=optarg;                                
                                break;                                          
			case 'V':
				conf.motionvidpipe=optarg;
				break;
			case 'w':
				conf.lightswitch=1;
				break;
                        case 'h':                                               
                        /* Do the same as an invallid option, but without the   
                         * 'invalid option' error.... */                        
                        default:                                                
                                usage();                                        
                                exit(1);                                       
                }                        
	optind=1;
}

void realconfig_cmdline (void)
{
	int c;

	opterr=0;
        while ((c=getopt(conf.argc, conf.argv, "-r:"))!=EOF) 
                switch (c) {
			case 'r':
				/* We should act like a 'child' and shouldn't
				   multiply :) */
				conf.nochild=1;
                                conf.realconfig=optarg;
                        default:
				break;
                }
	optind=1;
	opterr=1;
}

/*****************************************************************************/

void conf_cmdparse(char *cmd, char *arg1)
{
	int i = 0;

	if(!cmd)
		return;
  
	while( config_params[i].param_name != NULL ) {
		if(!strncasecmp(cmd, config_params[i].param_name , 255 + 50)) {
			if(config_params[i].conf_value && !arg1)
				return;
			config_params[i].copy( arg1, config_params[i].conf_value );
			return;
		}
		i++;
        }

	fprintf( stderr, "Unkown config option: \"%s\"\n", cmd ); 
        return;
}

void conf_process(FILE *fp)
{
	char line[256], *cmd = NULL, *arg1 = NULL;

	if (fp) {
		while (fgets(line, 256, fp) && configcontinue) {
			if(!(line[0]=='#' || strlen(line)< 2)) {/*skipcomment*/
				line[strlen(line) - 1] = 0; /*kill'\n'*/
				cmd = strtok( line, "\t =" );
				arg1 = strtok( NULL, "\t =");
				conf_cmdparse(cmd, arg1);
			}
		}
		fclose(fp);
	}
}


/**************************************************************************
    Main function, called from motion.c
*/
int conf_load (void)
{
	FILE *fp;
	char filename[2048]; // What is the maximum filelength anyways?

	if (!conf.nochild)
		realconfig_cmdline ();

	sprintf(filename, "%s/.motion/motion.conf", getenv("HOME"));
      	fp = fopen ("motion.conf", "r");
	if (!fp) {	/* specified file does not exist... try default file */
		fp = fopen (filename, "r");
		if (!fp) {
			fp = fopen ("/etc/motion.conf", "r");
			if (!fp) {	/* there is no config file.... use defaults */
				perror("no config file");
			}
		}
	}
	conf_process(fp);

	if (conf.realconfig) {
		fp = fopen (conf.realconfig, "r");
		if (!fp) {
			perror("realconfig file not found, exitting");
			exit(1);
		}
		conf_process(fp);
	}
	
	conf_cmdline();
	
	/* Prevent any additional childs from being created on SIGHUP */
	conf.nochild=1;
	return 0;
}

/************************************************************************
 * copy functions 
 *
 *   copy_bool   - convert a bool representation to int
 *   copy_int    - convert a string to int
 *   copy_string - just a string copy
 *   
 * @param str -   A char *, pointing to a string representation of the
 *                value.
 * @param value - points to the place where to store the value. 
************************************************************************/
static void copy_bool (char *str, void * val)
{
	if ( !strcmp(str, "1") || 
	     !strcasecmp(str, "yes") || 
	     !strcasecmp(str,"on")) 
	{
		*((int *)val) = 1;
	} else {
		*((int *)val) = 0;
	}
}

static void copy_int(char *str, void * val) {
	*((int *)val) = atoi(str);
}

static void copy_string(char *str, void * val) {
	*(char **)val=malloc(255);
	strncpy(*(char **)val, str, 255);	
}

static void config_realconfig(char *str, void *val)
{
	if (conf.nochild)
		return;

	if (conf.realconfig) {
		if (!fork()) {
			setpgrp();
			return;
		} else { 
			free(conf.realconfig);
		}
	}
	conf.realconfig=malloc(255);
	strncpy(conf.realconfig, str, 255);
}

static void usage (void)
{
        printf("motion Version "VERSION", Copyright 2000 Jeroen Vreeken\n");
        printf("\nusage:\tmotion [options]\n");
        printf("\n");
        printf("Overall settings:\n");
        printf("-C\t\tOutput changes count for every frame, usable for tuning\n");
        printf("-D\t\tDeamonize\n");
        printf("-a seconds\ttime between two automated snapshots, default: 0\n");
        printf("-c changes\tthreshold for detecting a change, default: %d\n", DEF_CHANGES);
	printf("-F file\t\tppm mask file to use as a motion filter.\n");
        printf("-f nr\t\tMaximum nr of frames per second, default: none\n");
	printf("-G sec\t\tMinimum gap between two shots in seconds, default: 0\n");
	printf("-L noise\tNoise level, default: %d\n", DEF_NOISELEVEL);
        printf("-l\t\tDo not locate and mark motion on output pictures\n");
        printf("-m\t\tOutput 'motion' images\n");
        printf("-N\t\tDon't output normal images\n");
	printf("-p\t\tOutput ppm images instead of jpeg, uses a lot of diskspace\n");
        printf("-t target dir\tdestination for snapshots.\n");
	printf("-Q\t\tBe quiet, don't output beeps when detecting motion\n");
	printf("-w\t\tActivate light switch filter\n");
        printf("\nDevice settings:\n");
	printf("-A ipnr\t\tAXIS camera ip address, default: not set\n");
        printf("-d device\tvideo4linux capture device, default: "VIDEO_DEVICE"\n");
        printf("-i input\tinput channel to use, default: not set\n");
        printf("-n norm\t\tnorm to use (pal/ntsc/secam), default: pal\n");
	printf("-P device\tvideo4linux video loopback input device, default: not set\n");
        printf("-q quality\tJPEG image quality, default: %d\n", DEF_QUALITY);
        printf("-s widthxheight\tPicture size, default: %dx%d\n", DEF_WIDTH, DEF_HEIGHT);
	printf("-V device\tvideo4linux video loopback input device for motion frames,\n");
	printf("\t\tdefault: not set.");
        printf("\nActions:\n");
        printf("-B\t\tEncode all jpg images to mpeg after an event using mpeg_encode\n");
        printf("-E command\tExecute 'command' when detecting motion.\n");
        printf("-M address\tSend a mail to 'address' when detecting motion.\n");
	printf("-O command\tExecute 'command' when saving an image.\n");
        printf("-S nr\t\tSend a SMS to nr using sms_client when detecting motion.\n");
        printf("-g seconds\tminimum gap between events, default %d\n", DEF_GAP);
        printf("\n");
}
                                                                                                                                                                                                                                                                                                                                                                                                                            
