/*	motion.c
 *
 *	Detect changes in a video stream.
 *	Copyright 2000 by Jeroen Vreeken (pe1rxq@amsat.org)
 *	This software is distributed under the GNU public license version 2
 *	See also the file 'COPYING'.
 *
 */
#include "motion.h"
#include "video.h"
#include "conf.h"
#include "alg.h"
#include "track.h"


// RPD:
#include <sys/types.h>
#include <sys/ipc.h>
#include <sys/sem.h>
#define MOTIONSEMKEY	0x045


extern struct config conf;

int deamon=0;
int snapshot=0;
int alarmtime=0;
int makemovie=0;
int finish=0;

int shots=0;
int event_nr=0, prev_event=0;
struct tm *currenttime=NULL;
time_t currenttimep, lasttime=0, eventtime=0, lastshot=0;
char file[255], filebase[255], timestamp[255], datestamp[255];
int lastrate=25;
int filesc=0, filesmc=0;
char **files=NULL, **filesm=NULL;
int *filesrate=NULL, *filesmrate=NULL;
int moved=0;

#ifdef HAVE_MYSQL
MYSQL *database=NULL;
#endif


void put_jpeg (FILE *picture, char *image, int width, int height, int quality)
{
        int y, x, line_width;
        JSAMPROW row_ptr[1];
        struct jpeg_compress_struct cjpeg;
        struct jpeg_error_mgr jerr;
        unsigned char *line;

        line = malloc (width * 3);
        if (!line)
                return;
        cjpeg.err = jpeg_std_error(&jerr);
        jpeg_create_compress (&cjpeg);
        cjpeg.image_width = width;
        cjpeg.image_height= height;
        cjpeg.input_components = 3;
        cjpeg.in_color_space = JCS_RGB;
        jpeg_set_defaults (&cjpeg);

        jpeg_set_quality (&cjpeg, quality, TRUE);
        cjpeg.dct_method = JDCT_FASTEST;
        jpeg_stdio_dest (&cjpeg, picture);

        jpeg_start_compress (&cjpeg, TRUE);

        row_ptr[0] = line;
        line_width = width * 3;
	for ( y = 0; y < height; y++) {
        for (x = 0; x < line_width; x+=3) {
                        line[x]   = image[x+2];
                        line[x+1] = image[x+1];
                        line[x+2] = image[x];
                }
                jpeg_write_scanlines (&cjpeg, row_ptr, 1);
                image += line_width;
        }
        jpeg_finish_compress (&cjpeg);
        jpeg_destroy_compress (&cjpeg);
        free (line);
}

void put_ppm (FILE *picture, char *image, int width, int height)
{
	int x, y;

	/*	ppm header
	 *	width height
	 *	maxval
	 */
	fprintf(picture, "P6\n");
	fprintf(picture, "%d %d\n", width, height);
	fprintf(picture, "%d\n", 255);
	for (y=0; y<height; y++) {
		for (x=0; x<width; x++) {
			/* ppm is rgb not bgr */
			fwrite(&image[y*width*3+x*3+2], 1, 1, picture);
			fwrite(&image[y*width*3+x*3+1], 1, 1, picture);
			fwrite(&image[y*width*3+x*3+0], 1, 1, picture);
		}
	}
}

char *get_ppm (FILE *picture, int width, int height)
{
	int x=0 ,y=0;
	char line[255];
	char *image;
	fgets(line, 255, picture);
	if (strncmp(line, "P6", 2)) {
		fprintf(stderr, "This is not a ppm file\n");
		return NULL;
	}
	/* skip comment */
	line[0]='#';
	while (line[0]=='#') fgets(line, 255, picture);
	/* check size */
	sscanf(line, "%d %d", &x, &y);
	if (x!=width || y!=height) {
		fprintf(stderr, "Wrong image size\n");
		return NULL;
	}
	/* Maximum value */
	fgets(line, 255, picture);
	sscanf(line, "%d", &x);
	if (x!=255) {
		fprintf(stderr, "Wrong image format\n");
		return NULL;
	}
	/* read data */
	image=malloc(width*height*3);
	for (y=0; y<height; y++) {
		for (x=0; x<width; x++) {
			/* ppm is rgb not bgr */
			fread(&image[y*width*3+x*3+2], 1, 1, picture);
			fread(&image[y*width*3+x*3+1], 1, 1, picture);
			fread(&image[y*width*3+x*3+0], 1, 1, picture);
		}
	}
	return image;
}

void put_picture (char *file, char *image, int width, int height)
{
	FILE *picture;

	picture=fopen(file, "w");
	if (conf.ppm)
		put_ppm (picture, image, conf.width, conf.height);
	else
		put_jpeg (picture, image, conf.width, conf.height, conf.quality);
	fclose(picture);
}

void send_sms(char *sms_nr, struct tm *currenttime)
{
	char message[150];

	if (!fork()) {
	    	printf("Sending SMS message to %s\n", sms_nr);
		sprintf(message, "Motion detected: %04d-%02d-%02d %02d:%02d:%02d\n",
		    currenttime->tm_year + 1900, currenttime->tm_mon +1,
		    currenttime->tm_mday, currenttime->tm_hour,
		    currenttime->tm_min, currenttime->tm_sec);
		execlp("sms_client", "sms_client", sms_nr, message, 0);
		perror("Unable to send SMS message");
		exit(0);
	}
	return;
}

void send_mail(char *mail_address, struct tm *currenttime)
{
	char message[150];
	int fdpipe[2];
	
	if (!fork()) {
	    	printf("Sending mail to %s\n", mail_address);
		sprintf(message, "This is an automated message generated by motion\n\n"
		    "Motion detected: %04d-%02d-%02d %02d:%02d:%02d\n"
		    ".\n",
		    currenttime->tm_year + 1900, currenttime->tm_mon +1,
		    currenttime->tm_mday, currenttime->tm_hour,
		    currenttime->tm_min, currenttime->tm_sec);
		pipe(fdpipe);
		if (!deamon) {
			close(0);
			dup(fdpipe[0]);
			close(fdpipe[0]);
		}
		write(fdpipe[1], message, strlen(message));
		close(fdpipe[1]);
		execlp("mail", "mail", mail_address, "-s", "MOTION ALERT", 0);
		perror("Unable to send message");
		exit(1);
	}
	return;
}

/* Execute 'command' with 'arg' as its argument.
   if !arg command is started with no arguments.
 */
void exec_command (char *command, char *arg)
{
	if (!fork()) {
		if (!arg) {
			printf("Executing external command '%s'\n", command);
			execlp(command, command, 0);
		} else {
			printf("Executing external command '%s %s'\n", command, arg);
			execlp(command, command, arg, 0);
		}
		perror("Unable to start external command");
		exit(1);
	}
	return;
}

void create_mpeg (int event_nr, int filesc, char **files, int *rate)
{
	int i, j;
	char params[255];
	char eventname[255];
	char mpegfilename[255];
	char *mpegname=eventname;
	FILE *paramfile=NULL;
	char *eventtime;

	if (!filesc) return;
	if (!fork()) {
		if (conf.oldlayout) {
			eventtime=files[0];
			while (*eventtime++!='-');
			strncpy(eventname, eventtime, 14);
		} else {
			strncpy(eventname, files[0], 11);/* daydir */
			strncpy(eventname+11, files[0]+11, 2);/* hour */
			strncpy(eventname+13, files[0]+14, 2);/* minute */
			strncpy(eventname+15, files[0]+17, 2);/* second */
		}
		if (files[0][strlen(files[0])-5]=='m') {
			eventname[17]='m';
			eventname[18]=0;
		} else eventname[17]=0;
		/* Cut of eventnr, we have te date so we don't need to sort */
		while((mpegname++)[0]!='-');
		sprintf(params, "%s.params", eventname);
		sprintf(mpegfilename, "%s.mpg", eventname);
		paramfile=fopen(params, "w");
		if(paramfile < 0) exit(0);
		printf("Starting mpeg encoding of event %d\n", event_nr);
		fprintf(paramfile,
			"PATTERN IBBPBBPBBPBBPBBP\n"
			"OUTPUT %s.mpg\n"
			"BASE_FILE_FORMAT ", eventname);
		if (!strncmp("ppm", files[0]+strlen(files[0])-3, 3))
			fprintf(paramfile, "PPM\n");
		else 
			fprintf(paramfile, "JPEG\n");
		fprintf(paramfile,
			"INPUT_CONVERT *\n"
			"GOP_SIZE 30\n"
			"SLICES_PER_FRAME 1\n"
			"INPUT_DIR .\n"
			"INPUT\n");
		for (i=0; i<filesc; i++) {
			if (!conf.adjustrate || rate[i]>25) {
				fprintf(paramfile, "%s\n", files[i]);
			} else {
				fprintf(stderr, "%d %d\n", rate[i], 25/(rate[i] ? rate[i]: 1));
				for (j=0; j<25/(rate[i] ? rate[i]: 2); j++) {
					fprintf(paramfile, "%s\n", files[i]);
					fflush(paramfile);
				}
			}
		}
		fprintf(paramfile,
			"END_INPUT\n"
			"PIXEL HALF\n"
			"RANGE 10\n"
			"PSEARCH_ALG LOGARITHMIC\n"
			"BSEARCH_ALG CROSS2\n"
			"IQSCALE 8\n"
			"PQSCALE 10\n"
			"BQSCALE 25\n"
			"FORCE_ENCODE_LAST_FRAME\n"
			"REFERENCE_FRAME ORIGINAL\n"/*DECODED*/
			"FRAME_RATE 25\n");
		fclose(paramfile);
		if (!fork()) {
			nice(5);
			execlp("mpeg_encode", "mpeg_encode", params, 0);
			perror("Failed to start mpeg_encode");
			exit(1);
		}
		if (wait(0)==-1) {
			perror("mpeg creation unsuccesfull");
			exit(1);
		}
		if (conf.onmpegcommand) {
			exec_command(conf.onmpegcommand, mpegfilename);
		}
		for (i=0; i<filesc; i++) {
			if (conf.jpg_cleanup) remove(files[i]);
			if (!conf.oldlayout) {
				strncpy(eventname, files[0], 16);
				eventname[16]=0;
				rmdir(eventname);
				strncpy(eventname, files[0], 13);
				eventname[13]=0;
				rmdir(eventname);
				strncpy(eventname, files[0], 10);
				eventname[10]=0;
				rmdir(eventname);
				strncpy(eventname, files[0], 7);
				eventname[7]=0;
				rmdir(eventname);
				strncpy(eventname, files[0], 4);
				eventname[4]=0;
				rmdir(eventname);
			}
			free(files[i]);
		}
		if (files)
			free(files);
		if (rate)
			free(rate);
		remove(params);
		exit(0);
	}
	return;
}

void mkdir_check(const char *pathname, mode_t mode)
{
	int value;
	
	value = mkdir(pathname, mode);
	if (value == 0) return;
	if (value == -1) {
		if (errno == EEXIST) return; else {
			printf("Error while creating directory");
			exit(0);
		}
	}
}

void mkfilebase(char *filebase, struct tm *currenttime)
{
	sprintf(filebase, "%04d/", currenttime->tm_year+1900);
	mkdir_check(filebase, 0755);
	sprintf(filebase, "%s%02d/", filebase, currenttime->tm_mon+1);
	mkdir_check(filebase, 0755);
	sprintf(filebase, "%s%02d/", filebase, currenttime->tm_mday);
	mkdir_check(filebase, 0755);
	sprintf(filebase, "%s%02d/", filebase, currenttime->tm_hour);
	mkdir_check(filebase, 0755);
	sprintf(filebase, "%s%02d/", filebase, currenttime->tm_min);
	mkdir_check(filebase, 0755);
}

#ifdef HAVE_MYSQL
void put_mysql (MYSQL *db, char *filebase, struct tm *currenttime, int type)
{
	char sqlquery[512];
	sprintf(sqlquery, "insert into security set filename='%s/%s', minute=%d, hour=%d, day=%d, month=%d, year=%d, type=%d", 
	    conf.filepath, filebase, 
	    currenttime->tm_min, currenttime->tm_hour, 
	    currenttime->tm_mday, currenttime->tm_mon+1, 
	    currenttime->tm_year+1900, type);
	if (mysql_query(db, sqlquery)) {
		printf("Error querying mysql");
		exit(0);
	}
}
#endif


/*	Our SIGNAL-Handler
 *	We need this to handle alarms and external signals
 */
static void sig_handler (int signo)
{
	switch(signo) {
		case SIGALRM: {
			/* Somebody (maybe we ourself) wants us to make a snapshot */
			snapshot=1;
			/* Set everything for the next snapshot */
			signal(SIGALRM, sig_handler);
			if (alarmtime) alarm(alarmtime);
			return;
		}
		case SIGUSR1: {
			/* Ouch! We have been hit from the outside! Someone wants us to
			   make a movie! */
			signal(SIGUSR1, sig_handler);
			makemovie=1;
			return;
		}
		case SIGTERM: {
			/* Somebody wants us to quit! We should better finish the actual
			    movie and end up! */
			signal(SIGTERM, sig_handler);
			makemovie=1;
			finish=1;
			return;
		}
		case SIGHUP: {
			/* Reread config files, also parse commandline again
			   because it still should override the config file */
			conf_load();
			printf("config reloaded\n");
			signal(SIGHUP, sig_handler);
			return;
		}
		default:
	}

	return;
}


void motion_detected (int diffs, int dev, int devpipe, int devmpipe, struct images *imgs)
{
	struct coord location;

	if (conf.locate || track.port)
		location=alg_locate_center(imgs->out, imgs->width, imgs->height);
	if (conf.locate)
		alg_locate(location, imgs->out, imgs->new, imgs->width, imgs->height);

	if (shots<conf.frame_limit-1 && currenttimep-lastshot >= conf.mingap ) {
		lastshot=currenttimep;
		/* Output the latest picture 'image_new' or image_out for motion picture.
		 */
		if (conf.motion_img || conf.new_img) {
			if (!conf.oldlayout) {
				mkfilebase (filebase, currenttime);
				sprintf(filebase, "%s%02d-%02d", filebase, currenttime->tm_sec, shots);
			} else {
				sprintf(filebase, "%02d-%04d%02d%02d%02d%02d%02d-%02d",
				    event_nr,
				    currenttime->tm_year + 1900,
				    currenttime->tm_mon +1,
				    currenttime->tm_mday,
				    currenttime->tm_hour,
				    currenttime->tm_min,
				    currenttime->tm_sec,
				    shots);
			}
		}
		if (conf.motion_img) {
			if (conf.ppm)
				sprintf(file, "%sm.ppm", filebase);
			else
				sprintf(file, "%sm.jpg", filebase);
			printf("Motion picture saved to: %s\n", file);
			filesmc++;
			filesm=realloc(filesm, filesmc*(sizeof(void *)));
			filesm[filesmc-1]=malloc(strlen(file)+1);
			strcpy(filesm[filesmc-1], file);
			filesmrate=realloc(filesmrate, filesmc*(sizeof(void *)));
			filesmrate[filesmc-1]=lastrate;
			draw_text(imgs->out, conf.width-70, conf.height-20, conf.height, conf.width, datestamp);
			draw_text(imgs->out, conf.width-70, conf.height-10, conf.height, conf.width, timestamp);
			put_picture(file, imgs->out, conf.width, conf.height);
			if (conf.onsavecommand) {
				exec_command(conf.onsavecommand, file);
			}
#ifdef HAVE_MYSQL
			if (conf.mysql_db)
				put_mysql(database, filebase, currenttime, 4);
#endif
			} 
		if (conf.new_img) {
			if (conf.ppm)
				sprintf(file, "%s.ppm", filebase);
			else
				sprintf(file, "%s.jpg", filebase);
			printf("Normal picture saved to: %s\n", file);
			filesc++;
			files=realloc(files, filesc*(sizeof(void *)));
			files[filesc-1]=malloc(strlen(file)+1);
			strcpy(files[filesc-1], file);
			filesrate=realloc(filesrate, filesc*(sizeof(void *)));
			filesrate[filesc-1]=lastrate;
			put_picture(file, imgs->new, conf.width, conf.height);
			if (conf.onsavecommand) {
				exec_command(conf.onsavecommand, file);
			}
#ifdef HAVE_MYSQL
			if (conf.mysql_db)
				put_mysql(database, filebase, currenttime, 1);
#endif
		}
	}
	if (!conf.quiet) printf("\a");
	printf("Motion detected %d\n", diffs);
	/* Take action if necesarry */
	if (event_nr!=prev_event) {
		if (conf.sms_nr) send_sms(conf.sms_nr, currenttime);
		if (conf.mail_address) send_mail(conf.mail_address, currenttime);
		if (conf.externcommand) exec_command(conf.externcommand, 0);
		prev_event=event_nr;
		eventtime=currenttimep;
	}
	lasttime=currenttimep;

	if (track.port) {
		if (track_move(dev, devpipe, devmpipe, location, imgs)) {
			moved=2;
		}
	}
}

int semid=-1;
//union semun semunarg;
int semunarg;

void grab_sem()
{
	struct sembuf sb = {0, -1, 0};  /* set to allocate resource */

//	printf("%d waiting for sem . . . \n", getpid()); fflush(stdout);
	if(semid >= 0) {
//		printf("%d got sem\n", getpid()); fflush(stdout);
		if (semop(semid, &sb, 1) == -1) {
		    perror("semop");
		    exit(1);
		}
	}
}

void release_sem()
{
	struct sembuf sb = {0, 1, 0};  /* set to free resource */

	if(semid >= 0) {
//		printf("%d releasing sem\n", getpid()); fflush(stdout);
		if (semop(semid, &sb, 1) == -1) {
		    perror("semop");
		    exit(1);
		}
//		printf("%d released sem\n", getpid()); fflush(stdout);
	}

}

int main (int argc, char **argv)
{
	int dev, devpipe=-1, devmpipe=-1, i, diffs;
	struct images imgs;
	time_t lastframe=0;
	char linkpath[255];
	FILE *picture;

	signal(SIGCHLD, SIG_IGN);

	conf.argv=argv;
	conf.argc=argc;
	conf_load();
	
	if (conf.mask_file) {
		if ((picture=fopen(conf.mask_file, "r"))) {
			imgs.mask=get_ppm(picture, conf.width, conf.height);
			fclose(picture);
		}
		if (!imgs.mask) {
			perror ("failed to read mask image");
			exit(1);
		}
	} else
		imgs.mask=NULL;
	
	if (conf.filepath) if (chdir(conf.filepath)) {
		perror("Unable to chdir to target directory");
		exit(1);
	}

	imgs.ref=malloc(conf.width*conf.height*3);
	imgs.out=malloc(conf.width*conf.height*3);

	//RPD:
	if(strcmp(conf.device, "/dev/video0") == 0) {

		printf("initializing semaphores\n");
		if ((semid = semget((key_t) MOTIONSEMKEY, 1, 0666 | IPC_CREAT)) == -1) {
		    perror("semget");
		    exit(1);
		}				        

		semunarg = 1;
		if (semctl(semid, 0, SETVAL, semunarg) == -1) {
		    perror("semctl");
		    exit(1);
		}
	} else {		// sem set is already created and initialized
		if ((semid = semget((key_t) MOTIONSEMKEY, 1, 0)) == -1) {
		    perror("semget");
		    exit(1);
		}
	}

	if (!conf.axis) {
		grab_sem();
		dev=open (conf.device, O_RDWR);
		if (dev < 0) {
			perror ("Failed to open video device");
			exit(1);
		}
		printf("Opened video device\n");
		release_sem();
	} else dev = 0;  // Shut the compiler up.
	/* set the device settings */
	imgs.new=vid_start (dev, conf.width, conf.height, conf.input, conf.norm);
	if (!imgs.new) {
		perror("Capture error");
		exit(1);
	}
	/* Allow videodevice to settle in */
	sleep(1);
	/* Capture first image, or we will get an alarm on start */
	if (!(imgs.new=vid_next(dev, imgs.new, conf.width, conf.height))) {
		perror("Capture error");
		exit(1);
	}
	/* create a reference frame */
	memcpy(imgs.ref, imgs.new, conf.width*conf.height*3);
	imgs.width=conf.width;
	imgs.height=conf.height;

	if (conf.vidpipe) {
		printf("Opening video loopback device for normal pictures\n");
		devpipe=vid_startpipe(conf.vidpipe, conf.width, conf.height);
		if (devpipe < 0) {
			printf ("Failed to open video loopback\n");
			exit(1);
		}
	}
	if (conf.motionvidpipe) {
		printf("Opening video loopback device for motion pictures\n");
		devmpipe=vid_startpipe(conf.motionvidpipe, conf.width, conf.height);
		if (devmpipe < 0) {
			printf ("Failed to open video loopback\n");
			exit(1);
		}
	}

#ifdef HAVE_MYSQL
	if(conf.mysql_db) {
		database=(MYSQL *) malloc(sizeof(MYSQL));
		mysql_init(database);
		if (!mysql_real_connect(database, conf.mysql_host, conf.mysql_user,
		    conf.mysql_password, conf.mysql_db, 0, NULL, 0)) {
			printf("Cannot connect to database\n");
			exit(1);
		}
	}
#endif	
	/* Deamonize */
	if (deamon) {
		printf("Going to deamon mode\n");
		close(0);
		close(1);
		close(2);
		if (fork()) exit(0);
	}

	/* Set signal handlers */
	signal(SIGALRM, sig_handler);
	signal(SIGUSR1, sig_handler);
	signal(SIGHUP, sig_handler);
	signal(SIGTERM, sig_handler);

	if (alarmtime) alarm(alarmtime);
	/* Should go on forever... unless you bought vaporware :) */
	while (finish ? makemovie : (int)(imgs.new=vid_next(dev, imgs.new, conf.width, conf.height))) {
		/* Make a differences picture in image_out */
		diffs=alg_diff(&imgs);
		if (conf.autobright)
			vid_autobright(dev, imgs.new, conf.width, conf.height);
		for (i=0; i<conf.width*conf.height*3; i++) {
			/* Old image slowly decays, this will make it even harder on
			 * a slow moving object to stay undetected
			 */
			imgs.ref[i]=(imgs.ref[i]+imgs.new[i])/2;
		}
#ifdef TRACKING
		if (track.port && moved) {
			for (i=0; i<conf.width*conf.height*3; i++)
				imgs.ref[i]=imgs.new[i];
			moved--;
			diffs=0;//continue;
		}
#endif
		currenttimep=time(NULL);
		currenttime=localtime(&currenttimep);
		if (lastframe!=currenttimep) {
			lastrate=shots;
			shots=-1;
			lastframe=currenttimep;
		}
		/* lights turned on ? */
		if (conf.lightswitch) {
			diffs=alg_lightswitch (diffs, dev, devpipe, devmpipe, &imgs);
		}
		/* Is the mpeg movie to long? */
		if (conf.maxmpegtime && (files || filesm))
			if (currenttimep-eventtime>= conf.maxmpegtime)
				makemovie=1;
		if ((currenttimep-lasttime>=conf.gap) || makemovie) {
			if (event_nr==prev_event || makemovie) {
				if (track.port) {
					printf("%d %d %d\n", prev_event, event_nr, makemovie);
					track_center();
					moved=3;
				}
				makemovie=0;
				event_nr++;
    				if (conf.mpeg) {
					if (filesc)
						create_mpeg(prev_event,
							    filesc,
							    files, 
							    filesrate
							    );
					if (filesmc)
						create_mpeg(prev_event,
							    filesmc, 
							    filesm, 
							    filesmrate
							    );
				}
				for (i=0; i<filesc; i++) free(files[i]);
				if (files)
					free(files);
				for (i=0; i<filesmc; i++) free(filesm[i]);
				if (filesm)
					free(filesm);
				if (filesrate)
					free(filesrate);
				if (filesmrate)
					free(filesmrate);
				filesc=0;
				files=NULL;
				filesmc=0;
				filesm=NULL;
				filesrate=NULL;
				filesmrate=NULL;
			}
		}
		shots++;
		sprintf(datestamp, "%04d-%02d-%02d",
		    currenttime->tm_year + 1900, currenttime->tm_mon + 1,
		    currenttime->tm_mday);
		sprintf(timestamp, "%02d:%02d:%02d-%02d",
		    currenttime->tm_hour, currenttime->tm_min,
		    currenttime->tm_sec, shots);
		draw_text(imgs.new, conf.width-70, conf.height-20, conf.height, conf.width, datestamp);
		draw_text(imgs.new, conf.width-70, conf.height-10, conf.height, conf.width, timestamp);

		/* The cat just walked in :) */
		if (diffs > conf.max_changes ) {
			motion_detected(diffs, dev, devpipe, devmpipe, &imgs);
		} else if (conf.always_diff) {
			printf("changes: %d\n", diffs);
		}

		if (snapshot) {
			if (!conf.snap_overwrite) {
				if (!conf.oldlayout) {
 					mkfilebase (filebase, currenttime);
					sprintf(filebase, "%s%02d-snapshot", filebase, currenttime->tm_sec);
				} else {
					sprintf(filebase, "%02d-%04d%02d%02d%02d%02d%02d-snapshot",
	    				    event_nr,
					    currenttime->tm_year + 1900,
					    currenttime->tm_mon +1,
					    currenttime->tm_mday,
					    currenttime->tm_hour,
					    currenttime->tm_min,
					    currenttime->tm_sec);
				}
			} else 
				sprintf(filebase, "lastsnap");
			if (conf.ppm)
				sprintf(file, "%s.ppm", filebase);
			else
				sprintf(file, "%s.jpg", filebase);
			printf("Snapshot saved to: %s\n", file);
			put_picture(file, imgs.new, conf.width, conf.height);
			snapshot=0;
			if (conf.onsavecommand) {
				exec_command(conf.onsavecommand, file);
			}
#ifdef HAVEMYSQL
			if (conf.mysql_db)
				put_mysql(database, filebase, currenttime, 2);
#endif
			if (!conf.snap_overwrite) {
				remove("lastsnap.jpg");
				remove("lastsnap.ppm");
				if (conf.ppm) {
//					sprintf(linkpath, "ln -s %s.ppm lastsnap.ppm", filebase);
					sprintf(linkpath, "%s.ppm", filebase);
					symlink(linkpath, "lastsnap.ppm");
				} else {
					sprintf(linkpath, "%s.jpg", filebase);
					symlink(linkpath, "lastsnap.jpg");
//					sprintf(linkpath, "ln -s %s.jpg lastsnap.jpg", filebase);
				}
//				system("rm -f lastsnap.jpg lastsnap.ppm");
//				system(linkpath);
			}
		}
		if (conf.vidpipe)
			vid_putpipe(devpipe, imgs.new, conf.width, conf.height);
		if (conf.motionvidpipe)
			vid_putpipe(devmpipe, imgs.out, conf.width, conf.height);
		/* This will limit the framerate to 1 frame while not detecting
		   motion.
		 */
		if (conf.low_cpu && lasttime!=currenttimep) {
			while(time(NULL)==currenttimep) usleep(500000);
		}
		fflush(0);
	}
	if (!finish)
		perror("Somebody stole the video device, lets hope we got his picture");
	close (dev);
	
	free(imgs.ref);
	free(imgs.out);
	exit(0);
}
