/*	video.c
 *
 *	Video stream functions for motion.
 *	Copyright 2000 by Jeroen Vreeken (pe1rxq@amsat.org)
 *	This software is distributed under the GNU public license version 2
 *	See also the file 'COPYING'.
 *
 */
#include "motion.h"
#include "video.h"
#include "conf.h"

/* for the v4l stuff: */
#include "sys/mman.h"
#include "sys/ioctl.h"
#include "linux/videodev.h"
/* for the axis stuff: */
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>

#define AXIS_WIDTH_1	320
#define AXIS_HEIGHT_1	240
#define AXIS_WIDTH_2	640
#define AXIS_HEIGHT_2	480
#define CONNECT_FAIL	1
#define DOWNLOAD_FAIL	2
#define DECOMPRESS_FAIL	4

extern struct config conf;

void grab_sem();
void release_sem();


/*******************************************************************************************
	Video4linux capture routines
*/

static int v4l_fmt=VIDEO_PALETTE_RGB24;
static int v4l_read_img=0;
static unsigned char *v4l_buffers[2];
static unsigned char *v4l_convert_buf; /* Used for converting yuv to bgr */
static int v4l_curbuffer=0;
static int v4l_maxbuffer=1;

static char *v4l_start (int dev, int width, int height, int input, int norm)
{
        struct video_capability vid_caps;
	struct video_channel vid_chnl;
	struct video_window vid_win;
	struct video_mbuf vid_buf;
	struct video_mmap vid_mmap;
	char *map;

	if (ioctl (dev, VIDIOCGCAP, &vid_caps) == -1) {
		perror ("ioctl (VIDIOCGCAP)");
		return (NULL);
	}
	if (vid_caps.type & VID_TYPE_MONOCHROME) v4l_fmt=VIDEO_PALETTE_GREY;
	/* Set channels and norms, NOT TESTED my philips cam doesn't have a
	 * tuner. update: it works with my brothers haupage wintv (bttv driver)
	 */
        if (input != IN_DEFAULT) {
                vid_chnl.channel = -1;
                if (ioctl (dev, VIDIOCGCHAN, &vid_chnl) == -1) {
                        perror ("ioctl (VIDIOCGCHAN)");
                } else {
                        vid_chnl.channel = input;
                        vid_chnl.norm    = norm;
                        if (ioctl (dev, VIDIOCSCHAN, &vid_chnl) == -1) {
                                perror ("ioctl (VIDIOCSCHAN)");
                                return (NULL);
                        }
                }
        }
	if (ioctl (dev, VIDIOCGMBUF, &vid_buf) == -1) {
		fprintf(stderr, "\tno mmap falling back on read\n");
		if (ioctl (dev, VIDIOCGWIN, &vid_win)== -1) {
			perror ("ioctl VIDIOCGWIN");
			return (NULL);
		}
		vid_win.width=width;
		vid_win.height=height;
		vid_win.clipcount=0;
		if (ioctl (dev, VIDIOCSWIN, &vid_win)== -1) {
			perror ("ioctl VIDIOCSWIN");
			return (NULL);
		}
		v4l_read_img=1;
		map=malloc(width*height*3);
		v4l_maxbuffer=1;
		v4l_curbuffer=0;
		v4l_buffers[0]=map;
		return (map);
	}
	/* If we are going to capture greyscale we need room to blow the image up */
	if (v4l_fmt==VIDEO_PALETTE_GREY) {
		map=mmap(0, vid_buf.size*3, PROT_READ|PROT_WRITE, MAP_SHARED, dev, 0);
		v4l_maxbuffer=1;
		v4l_buffers[0]=map;
	} else {
		map=mmap(0, vid_buf.size, PROT_READ|PROT_WRITE, MAP_SHARED, dev, 0);
		if (vid_buf.frames>1) {
			v4l_maxbuffer=2;
			v4l_buffers[0]=map;
			v4l_buffers[1]=map+vid_buf.offsets[1];
		} else {
			v4l_maxbuffer=1;
		}
	}

	if ((unsigned char *)-1 == (unsigned char *)map) {
		return (NULL);
	}
	v4l_curbuffer=0;
	vid_mmap.format=v4l_fmt;
	vid_mmap.frame=v4l_curbuffer;
	vid_mmap.width=width;
	vid_mmap.height=height;
	if (ioctl(dev, VIDIOCMCAPTURE, &vid_mmap) == -1) {
		printf("Failed with RGB, trying YUV420P palette\n");
		v4l_fmt=VIDEO_PALETTE_YUV420P;
		vid_mmap.format=v4l_fmt;
		/* Try again... */
		if (ioctl(dev, VIDIOCMCAPTURE, &vid_mmap) == -1) {
			perror("ioctl VIDIOCMCAPTURE");
			return (NULL);
		}
		v4l_convert_buf=malloc(width*height*3);
		if (!v4l_convert_buf) {
			perror("Failed to allocate video conversion buffer");
			return (NULL);
		}
	}
	return map;
}

static char *v4l_next (int dev, char *map, int width, int height)
{
	int i, tmp;
	int frame=v4l_curbuffer;
	unsigned char *grey, *rgb, *y, *u, *v, *r, *g, *b;
	struct video_mmap vid_mmap;

    	sigset_t    set, old;

	if (v4l_read_img) {
		if (v4l_fmt==VIDEO_PALETTE_GREY) {
			if (read(dev, map, width*height) != width*height)
				return NULL;
		} else {
			if (read(dev, map, width*height*3) != width*height*3)
				return NULL;
		}
	} else {
		vid_mmap.format=v4l_fmt;
		vid_mmap.width=width;
		vid_mmap.height=height;
    
        	sigemptyset (&set);			//BTTV hates signals during IOCTL
	        sigaddset (&set, SIGCHLD);		//block SIGCHLD & SIGALRM
       		sigaddset (&set, SIGALRM);		//SIGALRM
       		sigaddset (&set, SIGUSR1);		//SIGUSR1
       		sigaddset (&set, SIGTERM);		//SIGTERM
       		sigaddset (&set, SIGHUP);		//SIGHUP
        	sigprocmask (SIG_BLOCK, &set, &old);	// for the time we do capturing
                    
	// RPD:
	grab_sem();
		map=v4l_buffers[v4l_curbuffer];
		v4l_curbuffer++;
		if (v4l_curbuffer >= v4l_maxbuffer)
			v4l_curbuffer=0;
		vid_mmap.frame=v4l_curbuffer;
		if (ioctl(dev, VIDIOCMCAPTURE, &vid_mmap) == -1) {
	        	sigprocmask (SIG_UNBLOCK, &old, NULL);
			return (NULL);
		}

		vid_mmap.frame=frame;
		if (ioctl(dev, VIDIOCSYNC, &vid_mmap.frame) == -1) {
	        	sigprocmask (SIG_UNBLOCK, &old, NULL);
			return (NULL);
		}
		
	release_sem();
        	sigprocmask (SIG_UNBLOCK, &old, NULL);	//undo the signal blocking
	}
	/* Blow up a grey */
	if (v4l_fmt==VIDEO_PALETTE_GREY) {
		i=width*height;
		grey=map+i-1;
		rgb=map+i*3;
		for (; i>=0; i--, grey--) {
			*(rgb--)=*grey;
			*(rgb--)=*grey;
			*(rgb--)=*grey;
		}
	}
	/* YUV420 planar to RGB conversion: */
	if (v4l_fmt==VIDEO_PALETTE_YUV420P) {
		r=v4l_convert_buf+2;
		g=v4l_convert_buf+1;
		b=v4l_convert_buf;
		y=map;
		u=y+width*height;
		v=y+width*height+(width*height)/4;
		for (i=0; i<width*height; i++) {
			tmp=(((int)*y-16)*76284 + 104595*((int)*u-128)) >>16;
			if (tmp<0) tmp=0; if (tmp>255) tmp=255;
			*b=tmp;
			tmp=(((int)*y-16)*76284 - 53281*((int)*u-128) - 25625*((int)*v-128)) >>16;
			if (tmp<0) tmp=0; if (tmp>255) tmp=255;
			*g=tmp;
			tmp=(((int)*y-16)*76284 + 132252*((int)*v-128)) >>16;
			if (tmp<0) tmp=0; if (tmp>255) tmp=255;
			*r=tmp;
			if (i & 1) {
				u++;
				v++;
			}
			if (i%(width*2)==width-1) {
				u-=width/2;
				v-=width/2;
			}
			r+=3;
			g+=3;
			b+=3;
			y++;
		}
		map=v4l_convert_buf;
	}
	return map;
}

static int v4l_open_vidpipe(void)
{
	int pipe;
	FILE *vloopbacks;
	char pipepath[255];
	char buffer[255];
	char *loop;
	char *input;
	char *istatus;
	char *output;
	char *ostatus;
	
	vloopbacks=fopen("/proc/video/vloopback/vloopbacks", "r");
	if (!vloopbacks) {
		perror ("Failed to open '/proc/video/vloopback/vloopbacks");
		return -1;
	}
	/* Read vloopback version*/
	fgets(buffer, 255, vloopbacks);
	printf("\t%s", buffer);
	/* Read explaination line */
	fgets(buffer, 255, vloopbacks);
	while (fgets(buffer, 255, vloopbacks)) {
		if (strlen(buffer)>1) {
			buffer[strlen(buffer)-1]=0;
			loop=strtok(buffer, "\t");
			input=strtok(NULL, "\t");
			istatus=strtok(NULL, "\t");
			output=strtok(NULL, "\t");
			ostatus=strtok(NULL, "\t");
			if (istatus[0]=='-') {
				sprintf(pipepath, "/dev/%s", input);
				pipe=open(pipepath, O_RDWR);
				if (pipe>=0) {
					printf("\tInput: /dev/%s\n", input);
					printf("\tOutput: /dev/%s\n", output);
					return pipe;
				}
			}
		}
	}
	return -1;
}

static int v4l_startpipe (char *devname, int width, int height)
{
	int dev;
	struct video_picture vid_pic;
	struct video_window vid_win;

	if (!strcmp(devname, "-")) {
		dev=v4l_open_vidpipe();
	} else {
		dev=open(devname, O_RDWR);
	}
	if (dev < 0)
		return(-1);

	if (ioctl(dev, VIDIOCGPICT, &vid_pic)== -1) {
		perror("ioctl VIDIOCGPICT");
		return(-1);
	}
	vid_pic.palette=VIDEO_PALETTE_RGB24;
	if (ioctl(dev, VIDIOCSPICT, &vid_pic)== -1) {
		perror("ioctl VIDIOCSPICT");
		return(-1);
	}
	if (ioctl(dev, VIDIOCGWIN, &vid_win)== -1) {
		perror("ioctl VIDIOCGWIN");
		return(-1);
	}
	vid_win.height=height;
	vid_win.width=width;
	if (ioctl(dev, VIDIOCSWIN, &vid_win)== -1) {
		perror("ioctl VIDIOCSWIN");
		return(-1);
	}
	return dev;
}

static int v4l_putpipe (int dev, char *image, int height, int width)
{
	return write(dev, image, height*width*3);
}

static void v4l_autobright (int dev, unsigned char *image, int height, int width)
{
	struct video_picture vid_pic;
	int i, j=0, avg=0, offset=0;
	
	for (i=0; i<width*height*3; i+=100) {
		avg+=*image++;
		j++;
	}
	avg=avg/j;
	
	if (avg > 190 || avg < 64) {
		if (ioctl(dev, VIDIOCGPICT, &vid_pic)==-1) {
			perror("ioctl VIDIOCGPICT");
		}
		if (avg > 190) {
			offset=avg-190;
			if (vid_pic.brightness > 100*offset)
				vid_pic.brightness-=100*offset;
		}
		if (avg < 64) {
			offset=64-avg;
			if (vid_pic.brightness < 65535-100*offset)
				vid_pic.brightness+=100*offset;
		}
		printf("auto_brightness: %d\n", vid_pic.brightness);
		if (ioctl(dev, VIDIOCSPICT, &vid_pic)==-1) {
			perror("ioctl VIDIOCSPICT");
		}
	}
}

/*******************************************************************************************
	Axis 2100 capture routines
*/

/*  This is a driver for Motion version 1.6 for the Axis 2100 and similar cameras.
    Copyright (C) 2000  Tristan Willy

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/


static int axis_connect_http(void);
static FILE *axis_download_jpg(void);
static unsigned char *axis_decompress_jpeg();

/* Global Vars. */
static struct sockaddr_in axis_address;
static int axis_socketfd;
static char *axis_get_command;


/* API */


/* Things needed to be included in this function. 
 *    - Setup the IP structers
 *    - Check to make sure the width and height are correct
 *    - Ping to make sure the device is reachable. (NOT DONE YET)
 */ 
static char *axis_start (int dev, int width, int height, int input, int norm){
	char *pict_ptr;

	/* Just making sure... */
	if((width != AXIS_WIDTH_1 || height != AXIS_HEIGHT_1) && (width != AXIS_WIDTH_2 || height != AXIS_HEIGHT_2)){
		printf("Error: the width and height are incorrect. An abort is neccessary. ABORT...\n");
		return NULL;
	}

	/* Determine what CGI-Script is used for the resolution. */
	if((axis_get_command = malloc(45)) == NULL){  // Just a bit more than needed.
		perror("driver.c");
		return NULL;
	}
	
	if(width == AXIS_WIDTH_1)
		strcpy(axis_get_command, "GET /cgi-bin/fullsize.jpg HTTP/1.0\r\n\r\n");
	else
		strcpy(axis_get_command, "GET /cgi-bin/hugesize.jpg HTTP/1.0\r\n\r\n");

		
	/* Make enough memory, 3 bytes per pixel (RGB - 8bit) */
	if((pict_ptr = malloc(width * height * 3)) == NULL){
		perror("driver.c");
		return NULL;
	}

	/* Set up the address */
	axis_address.sin_family = AF_INET;
	axis_address.sin_addr.s_addr = inet_addr(conf.axis);
	axis_address.sin_port = htons(80);


	return pict_ptr;
}

/* Captures the next frame. MUST RETURN A FRAME OTHERWISE THE
 * PROGRAM EXITS! */
static char *axis_next(int dev, char *map, int width, int height){
	FILE *jpeg;
	unsigned char *uncompressed;
	int flag = 1;  // Start initial loop

	while(flag){
		flag = 0;

		/* Make the socket. */
		axis_socketfd = socket(AF_INET, SOCK_STREAM, 0);

		if(axis_connect_http()){
			close(axis_socketfd);
			flag++;
		}

		if((jpeg = axis_download_jpg()) == NULL && !flag){
			close(axis_socketfd);
			flag++;
		}  	
		
		/* Decompress the jpeg. */
		if((uncompressed = axis_decompress_jpeg(jpeg)) == NULL && !flag){
			fclose(jpeg);
			flag++;
		}
	}

	/* Now copy the image over. */
	memcpy(map, uncompressed, width * height * 3);

	/* Cleanup */
	free(uncompressed);
	fclose(jpeg);	

	return map;     /* a good memory location with a valid address */
}


/* COMPONENTS */

/* Connect and then release. */
static int axis_connect_http(){

	if((connect(axis_socketfd, (struct sockaddr *)&axis_address, sizeof(axis_address))) == -1){
		perror("Connect");
		return 1;
	}

	write(axis_socketfd, axis_get_command, strlen(axis_get_command));

	return 0;
}

/* Get information from the server until 3 newlines are passed.
 * After that then save the data to a file. 
 */
static FILE *axis_download_jpg(void){
	ssize_t ret, i = 0;
	char buff[512];
	FILE *jpeg;
	int newlines = 0;


	if((jpeg = tmpfile()) == NULL)
		return NULL;

	while((ret = read(axis_socketfd, buff, 512))){
		if(newlines < 3)
			for(i = 0; i < ret && newlines < 3; i++)
				 if(buff[i] == '\n')
				 	newlines++;
				 	
		write(fileno(jpeg), buff + i, ret - i);
		i = 0;
	}

	rewind(jpeg);
	close(axis_socketfd);
	return jpeg;
}


/* Decompress the jpeg and turn it into a BGR (1*1*1) pointer. */
static unsigned char *axis_decompress_jpeg(FILE *jpeg){
	struct jpeg_decompress_struct cinfo;
	struct jpeg_error_mgr jerr;
	unsigned char *line, *pic, *old_pic;
	JSAMPROW row[1];
	unsigned int i, line_size;

	cinfo.err = jpeg_std_error(&jerr);
	jpeg_create_decompress(&cinfo);

	jpeg_stdio_src(&cinfo, jpeg);

	jpeg_read_header(&cinfo, TRUE);

	cinfo.out_color_space = JCS_RGB;
	
	jpeg_start_decompress(&cinfo);
	line_size = cinfo.output_width * 3;


	if((line = malloc(cinfo.output_width * 3)) == NULL)
		return NULL;

	if((pic = malloc(cinfo.output_width * cinfo.output_height * 3)) == NULL){
		free(line);
		return NULL;
	}

	old_pic = pic;
	row[0] = line;
	while(cinfo.output_scanline < cinfo.output_height){
		jpeg_read_scanlines(&cinfo, row, 1);

		for(i = 0; i < line_size; i += 3){  // Inverse the image bytes.
			pic[i] = line[i+2];
			pic[i+1] = line[i+1];
			pic[i+2] = line[i];
		}
		pic += line_size;
	}

	jpeg_finish_decompress(&cinfo);
	jpeg_destroy_decompress(&cinfo);
	free(line);

	return old_pic;
}



/*******************************************************************************************
	Wrappers calling the actual capture routines
*/
char *vid_start (int dev, int width, int height, int input, int norm)
{
	if (!conf.axis) return v4l_start (dev, width, height, input, norm);
	else return axis_start (dev, width, height, input, norm);
}

char *vid_next (int dev, char *map, int width, int height)
{
	unsigned char *retval;

	if (!conf.axis) retval=v4l_next (dev, map, width, height);
	else retval= axis_next (dev, map, width, height);
	return retval;
}

int vid_startpipe (char *devname, int width, int height)
{
	return v4l_startpipe (devname, width, height);
}

int vid_putpipe (int dev, char *image, int height, int width)
{
	return v4l_putpipe (dev, image, height, width);
}

void vid_autobright (int dev, char *image, int height, int width)
{
	if (!conf.axis) 
		v4l_autobright(dev, image, height, width);
}

char *vid_keepalive(struct images *imgs, int dev, int pipe, int mpipe)
{
	imgs->new=vid_next(dev, imgs->new, imgs->width, imgs->height);
	if (imgs->new) {
		memcpy(imgs->ref, imgs->new, imgs->height*imgs->width*3);
		if (conf.vidpipe)
			vid_putpipe(pipe, imgs->new, imgs->width, imgs->height);
		if (conf.motionvidpipe)
			vid_putpipe(mpipe, imgs->out, imgs->width, imgs->height);
	}
	return imgs->new;
}
