/*
 *  Copyright (C) 2006 Simon Funk - simonfunk@gmail.com
 *  
 *  This program is free software; you can redistribute it and/or modify it under 
 *  the terms of the GNU General Public License as published by the Free Software 
 *  Foundation; either version 2 of the License, or (at your option) any later 
 *  version.
 *  
 *  This program is distributed in the hope that it will be useful, but WITHOUT ANY 
 *  WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A 
 *  PARTICULAR PURPOSE. See the GNU General Public License for more details.
 */
//#include	"config.h"

#include "Aparse.h"
#include "Stream.h"
#include <stdarg.h>

/*
 * Does wonderful things to parse your argv
 *   list.  Everything successfully parsed
 *   will be removed from the argv array.
 * If the program is envoked with -help,
 *   the Helpfunc will be called with argv as
 *   it's one parameter, and the program will
 *   exit with status 0.
 * If the program is envoked with -use, the
 *   usage will be printed, and the program will
 *   exit with status 0.
 * If an error in the parameter list is detected,
 *   the usage will be printed, and the program
 *   will exit with a non-0 exit status.
 *
 * Use: aparse(argv, Helpfunc,
 *                ControlString1, variables...,
 *                ControlString2, variables...,
 *                ...,
 *                NULL);
 *
 * Where:
 *    argv            is exactly as it is passed to main.
 *    Helpfunc        is a pointer to a help function, or NULL
 *                    Helpfunc will be called with argv as a parameter.
 *    ControlString    is a cstring who's format is explained below
 *    variables...    are pointers to variables corresponding to
 *                    entries in the ControlString.
 *
 * Example:
 *  aparse(argv,NULL,
 *                "in=%f(InputFile,-)%i<0,1>",&infile,&inspecified,
 *                "out=%f(OutputFile,-)",&outfile,
 *                "-fast%i<1,0>",&fastflag,
 *                "color=%r(R,1.)%r(G,1.)%r(B,1.)",&r,&g,&b,
 *                NULL);
 *
 * Each control string matches one parameter.
 *
 * Control string format:
 *
 *    %s                means return a cstring (via a char **, NOT a char *)
 *    %c                means return a character (via an int *, NOT a char *)
 *    %f                like %s, but '-' returns NULL.  Use for Filenames.
 *    %i                means return an int (via an int *)
 *    %r                means return a real (via a real *)
 *    (name,default)    encloses a 'USE' name and default value for a
 *                      parameter.
 *     <on,off>         encloses values to assign to the matching variable
 *                      depending on whether or not the given keyword
 *                      was matched.
 *    ...               a triple dot is a special case that means it is
 *                      ok for there to be unparsed arguments left over.
 *
 * As any non-keyworded entry will match the first not-already-matched
 * arg in argv, it is a good idea to list all such entries AFTER all
 * keyworded entries.
 *
 * Additional parameter types (eg cstring, file, int, etc..) can be added
 * by making a call to AparseAddParser(key,parsefunc); before calling
 * Aparse itself.  Here, "key" is a char specifying the type indicator
 * (eg, 'i' for integer) and "parsefunc" is a function taking two parameters:
 *    int parsefunc(cstring value, pointer variable);
 * Parsefunc extracts the value from "value" leaving the result in *variable.
 * Prasefunc returns 1 on error, and 0 on success.  See AparseParseInt,
 * AparseParseReal, etc... for examples.  Also see AparseInitFuncs();
 */

#define MaxFuncs 50

    static int				AparseChar[MaxFuncs];
    static ParseFunctionPtr	AparseParser[MaxFuncs];
    static int				AparseNumFuncs = 0;

#define MaxArgs 100

    static cstring AparseUse[MaxArgs];
    static int    AparseNumArgs = 0;

#define FailNot 0
#define FailUse 1
#define FailHelp 2
#define FailFail 3

void AparseInitFuncs(void);
void AparseParseSubparm(Stream cstream,Stream ustream,Stream astream,
						int vartype,pointer varptr,
						int argn,int *subargn,cstring keyword,int *fail);
void AparseParseBoolean	(Stream cstream,Stream ustream,Stream astream,
					 int vartype,pointer varptr,
					 int argn,int *subargn,cstring keyword,int *fail);
void AparseParseValue	(Stream cstream,Stream ustream,Stream astream,
						 int vartype,pointer varptr,
						 int argn,int *subargn,cstring keyword,int *fail);
void AparseExit(char **argv,int code,HelpfuncPtr hf);

int		AparseParseFile		(cstring str,cstring *ptr);
int		AparseParseString	(cstring str,cstring *ptr);
int		AparseParseChar		(cstring str,int *ptr);
int		AparseParseInt		(cstring str,int *ptr);
int		AparseParseReal		(cstring str,real *ptr);

int		AparseParseParam	(cstring parm,int vartype,pointer varptr);
void	AparseInitFuncs		(void);
char*	AparseExtractArg	(cstring argv[],cstring keyword);

int Aparse(cstring* argv, HelpfuncPtr help, ...)
{
    va_list pvar;
    cstring control;
    int fail;
    int extra_ok;    /* is it ok for there to be stuff left over? */

    if (AparseNumFuncs==0)
        AparseInitFuncs();

	va_start(pvar, help);

    if (argv[1] && strcmp(argv[1],"-use")==0)
        fail = FailUse;
    else if (argv[1] && strcmp(argv[1],"-help")==0)
        fail = FailHelp;
    else
        fail = FailNot;

	extra_ok = 0;

    AparseNumArgs = 0;

    while ((control = va_arg(pvar, cstring)) != NULL) {

        Stream cstream,ustream,astream;
        cstring arg,keyword;
        char use[100];
        int c;
        int subparm;

		if (AparseNumArgs >= MaxArgs) {
			fprintf(ErLogFile(),"Too many control params %d. Max %d.\n",
                    AparseNumArgs, MaxArgs);
			fprintf(ErLogFile(),"Ignoring the rest.  Better increase MaxArgs.\n");
			break;
		}

		if (!strcmp(control,"...")) {
			extra_ok = 1;
			continue;
		}

        use[0] = EOS;

        cstream = StreamOpenString(control);
        ustream = StreamOpenString(use);

        keyword = StreamReadUntil(cstream,"%");

        StreamWriteString(ustream,keyword);

        astream = NULL;

        if (!fail) {
            arg = AparseExtractArg(argv,keyword);
            if (arg)
                astream = StreamOpenString(arg);
        }

        subparm = 1;
        while ((c=StreamGetc(cstream)) != EOS) {

            int vartype;
            pointer varptr;

            if (c!='%') {
                fprintf(ErLogFile(),"Aparse control error in arg %d.%d (%s):\n",
                    AparseNumArgs+1,subparm,keyword);
                fprintf(ErLogFile(),"Got '%c' when expecting '%%'\n",c);
                exit(2);
            }

            vartype = StreamGetc(cstream);
            varptr  = va_arg(pvar, pointer);

            AparseParseSubparm(
                        cstream,ustream,astream,
                        vartype,varptr,
                        AparseNumArgs+1,&subparm,
                        keyword,&fail);
        }

        free(keyword);
        StreamClose(ustream);
        StreamClose(cstream);
        if (astream)
            StreamClose(astream);
        AparseUse[AparseNumArgs++] = crbncpy(use);
    }

    va_end(pvar);

    if (!fail && argv[1] && !extra_ok) {
        fprintf(ErLogFile(),"What am I supposed to do with '%s'?\n",argv[1]);
        fail = FailFail;
    }

    if (fail)
        AparseExit(argv,fail,help);
    else {
        int i;

        for (i=0; i<AparseNumArgs; i++)
            free(AparseUse[i]);
    }

    return(0);
}

void AparseParseSubparm(Stream cstream,Stream ustream,Stream astream,
						int vartype,pointer varptr,
						int argn,int *subargn,cstring keyword,int *fail)
{
    int c;

    c = StreamGetc(cstream);

    switch (c) {

    case '<':

        AparseParseBoolean(
                    cstream,ustream,astream,
                    vartype,varptr,
                    argn,subargn,keyword,fail);
        break;

    case '(':
    case '%':

        StreamUngetc(cstream,c);
        AparseParseValue(
                    cstream,ustream,astream,
                    vartype,varptr,
                    argn,subargn,keyword,fail);
        break;

    default:

        fprintf(ErLogFile(),"Aparse error: got '%c' when expecting <,(, or %%\n",
                c);
        exit(2);
    }
}


void AparseParseBoolean	(Stream cstream,Stream ustream,Stream astream,
						 int vartype,pointer varptr,
						 int argn,int *subargn,cstring keyword,int *fail)
{
    cstring onval,offval,arg,u;

    onval = StreamReadUntil(cstream,",");
    StreamGetc(cstream);

    offval = StreamReadUntil(cstream,",>");

    if (StreamGetc(cstream) == ',') {
        u = StreamReadUntil(cstream,">");
        StreamGetc(cstream);
        StreamWriteString(ustream," (means ");
        StreamWriteString(ustream,u);
        StreamWriteString(ustream,")");
        free(u);
    }

    if (*fail)
        return;

    if (astream)
        arg = onval;
    else
        arg = offval;

    if (AparseParseParam(arg,vartype,varptr)) {

                fprintf(ErLogFile(),"Aparse control error in arg %d (%s):\n",
                    argn,keyword);
                fprintf(ErLogFile(),"Can't parse '%s' with the '%c' control.\n",
                    arg,vartype);
                exit(2);
    }
    free(onval);
    free(offval);
}

void AparseParseValue	(Stream cstream,Stream ustream,Stream astream,
						 int vartype,pointer varptr,
						 int argn,int *subargn,cstring keyword,int *fail)
{
    cstring deflt;
    int c;

    if (*subargn > 1)
        StreamWriteString(ustream,",");

    if ((c = StreamGetc(cstream)) == '(') {

        /*
         * Extract the default value, and print
         * the usename into the use string.
         * The default (deflt) may be NULL.
         */

        cstring usename;

        usename = StreamReadUntil(cstream,",)");
        StreamWriteString(ustream,usename);
        free(usename);

        if ((c = StreamGetc(cstream)) == ')') {

            deflt = NULL;

        } else {

            if (c != ',') {
                fprintf(ErLogFile(),"Aparse control error in arg %d.%d (%s):\n",
                    argn,*subargn,keyword);
                fprintf(ErLogFile(),"Got '%c' when expecting ',' or ')'\n",c);
                exit(2);
            }

            deflt = StreamReadUntil(cstream,")");
            StreamGetc(cstream);
            StreamWriteString(ustream,"(");
            StreamWriteString(ustream,deflt);
            StreamWriteString(ustream,")");
        }

    } else {
        StreamUngetc(cstream,c);
        StreamWriteString(ustream,"???");
        deflt = NULL;
    }

    /*
     * Now extract the argument, or use the default.
     */
    if (!*fail) {

        if (astream) {

            cstring parm;

            parm = StreamReadUntil(astream,",");
            StreamGetc(astream);

            if (AparseParseParam(parm,vartype,varptr)) {

                (*fail)++;

                if (*keyword)
                 fprintf(ErLogFile(),
                  "Entry '%s' is not valid as parameter %d of %s.\n",
                  parm,*subargn,keyword);
                else
                 fprintf(ErLogFile(),
                  "Entry '%s' is not valid as parameter %d.%d\n",
                    parm,argn,*subargn);
            }

            free(parm);

        } else if (deflt) {

            if (AparseParseParam(deflt,vartype,varptr)) {

                (*fail)++;

                if (*keyword)
                 fprintf(ErLogFile(),
                  "Default '%s' is not valid as parameter %d of %s.\n",
                  deflt,*subargn,keyword);
                else
                 fprintf(ErLogFile(),
                  "Default '%s' is not valid as parameter %d.%d\n",
                    deflt,argn,*subargn);
            }

        } else {
            if (*keyword)
                fprintf(ErLogFile(),"Missing argument %d to %s.\n",
                    *subargn,keyword);
            else
                fprintf(ErLogFile(),"Missing argument!");
            (*fail)++;
        }
    }

    if (deflt)
        free(deflt);

    (*subargn)++;
}

void AparseExit(char **argv,int code,HelpfuncPtr hf)
{
    int i;

    if (code == FailHelp) {

        if (hf)
            (*hf)(argv);
        else
            fprintf(ErLogFile(),"Sorry, no additional help available.\n");

        exit(0);
    }

    fprintf(ErLogFile(),"\nUse: %s\n",argv[0]);

    for (i=0; i<AparseNumArgs; i++)
        fprintf(ErLogFile(),"\t%s\n",AparseUse[i]);

    fprintf(ErLogFile(),"\n");
    if (hf)
        fprintf(ErLogFile()," or: %s -help\n",argv[0]);

    fprintf(ErLogFile()," or: %s -use\n",argv[0]);

    if (code==FailFail)
        exit(1);
    else
        exit(0);
}


int AparseParseFile(cstring str,cstring *ptr)
{
    if (!strcmp(str,"-"))
        *ptr = NULL;
    else
        *ptr = crbncpy(str);
    return(0);
}

int AparseParseString(cstring str,cstring *ptr)
{
    *ptr = crbncpy(str);
    return(0);
}

int AparseParseChar(cstring str,int *ptr)
{
	*ptr = *str;	/* Intentionally allows for '\0' */
    return(0);
}

int AparseParseInt(cstring str,int *ptr)
{
    if (sscanf(str,"%d",ptr)!=1)
        return(1);
    return(0);
}

int AparseParseReal(cstring str,real *ptr)
{
    if (sscanf(str,"%lf",ptr)!=1)
        return(1);
    return(0);
}

void AparseAddParser(int c,ParseFunctionPtr f)
{
    AparseChar  [AparseNumFuncs] = c;
    AparseParser[AparseNumFuncs] = f;

    AparseNumFuncs++;
}

int AparseParseParam(cstring parm,int vartype,pointer varptr)
{
    int i;

    for (i=0; i<AparseNumFuncs; i++)
        if (AparseChar[i] == vartype)
            return((*AparseParser[i])(parm,varptr));

    fprintf(ErLogFile(),"Aparse error, unknown control: %%%c\n",vartype);
    exit(2);
    return(1);
}

void AparseInitFuncs(void)
{
    AparseAddParser('i',(ParseFunctionPtr)AparseParseInt);
    AparseAddParser('c',(ParseFunctionPtr)AparseParseChar);
    AparseAddParser('r',(ParseFunctionPtr)AparseParseReal);
    AparseAddParser('s',(ParseFunctionPtr)AparseParseString);
    AparseAddParser('f',(ParseFunctionPtr)AparseParseFile);
}

/*
 * Finds the first arg in argv beginning with
 * Keyword, and extracts (removes) it from the argv[],
 * returning a pointer to the rest of the arg.
 * (everything AFTER keyword).
 * Returns NULL if no match.
 * If keyword is "", the first arg will match.
 * Argv is assumed to start with the program name,
 * which will be skipped over. (Ie. argv[0])
 */
char *AparseExtractArg(cstring argv[],cstring keyword)
{
    char *cp;
    int i;

    for (argv++; *argv; argv++) {

        if (!strsame(keyword,*argv))
            continue;

        cp = argv[0]+strlen(keyword);

        i = 0;

        do {
            argv[i] = argv[i+1];
        } while (argv[i++]);

        return(cp);
    }

    return(NULL);
}


