/*
 *  Copyright (C) 2006 Simon Funk - simonfunk@gmail.com
 *  
 *  This program is free software; you can redistribute it and/or modify it under 
 *  the terms of the GNU General Public License as published by the Free Software 
 *  Foundation; either version 2 of the License, or (at your option) any later 
 *  version.
 *  
 *  This program is distributed in the hope that it will be useful, but WITHOUT ANY 
 *  WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A 
 *  PARTICULAR PURPOSE. See the GNU General Public License for more details.
 */

#include "Stream.h"

#define MaxStringLength 2000

static inline
Stream StreamCreateHalfBaked(int type, cstring name)
{
	Stream s;

	s = stalloc(StreamStruct);

	s->type   = type;
	s->name   = name;
	s->lineno = 1;
	s->refs   = 1;

	return s;
}

Stream StreamOpenDynamic2(cstring init, int append)
{
	Stream s;

	s = StreamCreateHalfBaked(StreamDynamic, "<Dynamic>");

	if (init) {
		s->u.dyn.buf    = crbncpy(init);
		s->u.dyn.len    = strlen(init);
		s->u.dyn.maxLen = s->u.dyn.len;
		s->u.dyn.at     = append?s->u.dyn.len:0;
	} else {
		s->u.dyn.len    = 0;
		s->u.dyn.at     = 0;
		s->u.dyn.maxLen = 255;
		s->u.dyn.buf    = zalloc(char, s->u.dyn.maxLen+1);
		s->u.dyn.buf[0] = '\0';
	}

	return s;
}

Stream StreamOpenDynamic(void)
{
	return StreamOpenDynamic2(NULL, 0);
}

static inline
int StreamIsDynamic(Stream s)
{
	if (!s)
		return 0;

	if (s->type == StreamDynamic)
		return 1;

	ErCreate("This is not a Dynamic Stream!");
	return 0;
}

void StreamDynamicRewind(Stream s)
{
	if (!StreamIsDynamic(s))
		return;

	s->u.dyn.at = 0;
}

cstring StreamDynamicString(Stream s)
{
	if (!StreamIsDynamic(s))
		return;

	return crbncpy(s->u.dyn.buf);
}

void StreamDynamicGrow(Stream s)
{
	cstring newBuf;
	int newLen;

	if (!StreamIsDynamic(s))
		return;

	newLen = (s->u.dyn.maxLen+1) * 2;
	newBuf = zalloc(char, newLen);
	strcpy(newBuf, s->u.dyn.buf);

	free(s->u.dyn.buf);
	s->u.dyn.buf    = newBuf;
	s->u.dyn.maxLen = newLen-1;
}


Stream StreamOpenString(cstring str)
{
	Stream s;

	s = StreamCreateHalfBaked(StreamString, "<String>");

	s->u.string.at    = str;
	s->u.string.start = str;

	return s;
}

Stream StreamOpenFl2(File fl, cstring nm, int closeOnClose)
{
	Stream s;

	if (!fl)
		return NULL;

	s = StreamCreateHalfBaked(StreamFile, crbncpy(nm));

	s->u.file.fp           = fl;
	s->u.file.closeOnClose = closeOnClose;

	return s;
}


Stream StreamAddRef(Stream s)
{
	if (s)
		s->refs++;

	return s;
}

void StreamDelRef(Stream s)
{
	if (!s)
		return;

	if (--(s->refs) > 0)
		return;

	switch(s->type) {
	
	case StreamFile:
		if (s->name)
			free(s->name);

		if (s->u.file.closeOnClose == 1)
			fclose(s->u.file.fp);
		else if (s->u.file.closeOnClose == 2)
			pclose(s->u.file.fp);
		break;

	case StreamString:
		// Don't need do clean anything special up...
		break;

	case StreamDynamic:
		free(s->u.dyn.buf);
		break;

	default:
		ErCreate("StreamDelRef: bogus Stream type.");
		return;
	}

	free(s);
}


Stream StreamOpenFl(File fl)
{
	return StreamOpenFl2(fl, NULL, 0);
}

Stream StreamOpenFile(cstring nm, cstring mode)
{
	File fl;

	if (!nm || !mode)
		return NULL;

	if (!(fl = fopen(nm, mode)))
		return NULL;

	return StreamOpenFl2(fl, nm, 1);
}

Stream StreamOpenFileRead(cstring nm)
{
	Stream s;

	if (nm) {
		if (!strcmp(filename_extension(nm), "gz")) {
			char buf[1024];	// Hazard note...
			File fl;
			sprintf(buf, "zcat \"%s\"", nm);
			fl = popen(buf, "r");
			if (!fl) {
				perror(buf);
				exit(1);
			}
			s = StreamOpenFl2(fl, nm, 2);
		} else if (!strcmp(filename_extension(nm), "gen")) {
			File fl;
			fl = popen(nm, "r");
			if (!fl) {
				perror(nm);
				exit(1);
			}
			s = StreamOpenFl2(fl, nm, 2);
		} else {
			s = StreamOpenFile(nm, "r");
		}
	} else {
		s = StreamOpenFl2(stdin, nm = "<stdin>", 0);
	}

	if (!s) {
		perror(nm);
		exit(1);
	}
	return s;
}

Stream StreamOpenFileWrite(cstring nm)
{
	Stream s;

	if (nm) {
		if (strcmp(filename_extension(nm), ".gz")) {
			s = StreamOpenFile(nm, "w");
		} else {
			ErFatal("Unimplemented.");
		}
	} else {
		s = StreamOpenFl2(stdout, nm = "<stdout>", 0);
	}

	if (!s) {
		perror(nm);
		exit(1);
	}

	return s;
}

void StreamClose(Stream s)
{
	StreamDelRef(s);
}


/*
 * Reads from stream s until one
 * of the characters in stops, or
 * EOS, is encountered.  The terminating
 * character is pushed back onto the stream,
 * and a fresh copy of the string thus far
 * read is returned.
 */
cstring StreamReadUntil(Stream s,cstring stops)
{
	char buf[MaxStringLength];
	int i,c;

	for (i=0; i<MaxStringLength-1; i++) {

		c = StreamGetc(s);

		if (c==EOS || isoneof(c,stops)) {

			StreamUngetc(s,c);
			buf[i] = EOS;
			return(crbncpy(buf));
		}

		buf[i] = c;
	}

	/*
	 * Potential badness.
	 */
	buf[i] = EOS;
	return(crbncpy(buf));
}

/*
 * Reads from stream s until the input
 * character is not one of the characters
 * in 'set', or EOS, is encountered.
 * The terminating
 * character is pushed back onto the stream,
 * and a fresh copy of the string thus far
 * read is returned.
 */
cstring StreamReadWhile(Stream s,cstring set)
{
	char buf[MaxStringLength];
	int i,c;

	for (i=0; i<MaxStringLength-1; i++) {

		c = StreamGetc(s);

		if (c==EOS || !isoneof(c,set)) {

			StreamUngetc(s,c);
			buf[i] = EOS;
			return(crbncpy(buf));
		}

		buf[i] = c;
	}

	/*
	 * Potential badness.
	 */
	buf[i] = EOS;
	return(crbncpy(buf));
}

void StreamWriteString(Stream s, cstring str)
{
	for (; *str; str++)
		StreamPutc(s,*str);
}

void StreamPutcn(Stream to, int c, int n)
{
	for (; n>0; n--)
		StreamPutc(to, c);
}

