/*
 *  Copyright (C) 2006 Simon Funk - simonfunk@gmail.com
 *  
 *  This program is free software; you can redistribute it and/or modify it under 
 *  the terms of the GNU General Public License as published by the Free Software 
 *  Foundation; either version 2 of the License, or (at your option) any later 
 *  version.
 *  
 *  This program is distributed in the hope that it will be useful, but WITHOUT ANY 
 *  WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A 
 *  PARTICULAR PURPOSE. See the GNU General Public License for more details.
 */

#ifndef STREAM
#define STREAM
#include "Toybox.h"

    /*
     * Stream io in a more general fassion.
	 *
	 * Can "open" strings, files, and now pipes!
     *
     * End of file is 0, not -1.
     *
     * Note the name in the case of a FILE Stream is
     * crbncpy'd on creation, but NOT free'd on closing.
     * This is to allow references to the filename to be
     * kept after closing the file.
     * It is assumed that the number of file opens in any
     * application will be small enough that this will not
     * use a noticeable amount of memory.
     */

#define EOS ('\0')

    typedef struct StreamStruct *Stream;

    struct StreamStruct {

        int     type;   // StreamFile, StreamString, etc..
        cstring name;   // Crbncpy'd if file, static if string.
        int     lineno; // Not supported in strings currently.
		int     refs;	// Standard GC.

        union {

            struct {
				int closeOnClose;	// 1=Close fp when the stream goes away; 2=pclose; 0=N/A
				FILE *fp; 
			} file;

            struct {
				cstring at;
				cstring start;
			} string;

			// Don't access this directly!  Implementation may change.
            struct {
				cstring buf;
				int     at;		// Current pointer within string.
				int     len;	// Current end of string (always points to '\0')
				int     maxLen;	// The buf is allocated to this + 1.
			} dyn;
        } u;
    };

#define StreamFile    0
#define StreamString  1
#define StreamDynamic 2

	Stream StreamOpenFl (File fl);
	Stream StreamOpenFl2(File fl, cstring nm, int closeOnClose);

	/*
	 * This opens a file in the specified mode:
	 */
	Stream StreamOpenFile(cstring nm, cstring mode);

	/*
	 * These are like the above, but if nm is NULL,
	 *  stdin or stdout are automatically selected,
	 *  and if nm ends in ".gz", zcat or gzip are
	 *  popened.  Also, beware these simply exit on
	 *  failure, rather than returning NULL.
	 */
	Stream StreamOpenFileWrite(cstring nm);
	Stream StreamOpenFileRead (cstring nm);

	/*
	 * Use this for reading from strings:
	 */
	Stream StreamOpenString(cstring str);

	/*
	 * Use this for reading or writing strings.
	 *
	 * The stream is opened on a COPY of init,
	 *  or on a blank string if init is NULL.
	 *
	 * If append is true, the pointer is started
	 *  at the end.
	 */
	Stream StreamOpenDynamic (void);
	Stream StreamOpenDynamic2(cstring init, int append);


	Stream StreamAddRef(Stream s);
	void   StreamDelRef(Stream s);
	void   StreamClose (Stream s);	// Deprecated; maps to DelRef.

	cstring StreamReadUntil  (Stream s, cstring stops);
	cstring StreamReadWhile  (Stream s, cstring set);
	void    StreamWriteString(Stream s, cstring str);

	/*
	 * This calls StreamPutc N times with c.
	 */
	void StreamPutcn(Stream to, int c, int n);

	/*
	 * Methods specific to Dynamic streams:
	 */
	void    StreamDynamicRewind(Stream s);	// Rewinds to the beginning.
	cstring StreamDynamicString(Stream s);	// Returns a crbncpy of the contents.

	void StreamDynamicGrow(Stream s);	// Internal; may change...

static inline
int StreamGetc(Stream s)
{
	int c;

	if (!s)
		return EOS;

	switch(s->type) {

	case StreamFile:
		while ((c = getc(s->u.file.fp)) == 0)	// Discard 0's as they will look like EOS's
			;
		if (c == EOF)
			c = EOS;
		break;

	case StreamString:
		if ((c = *(s->u.string.at)))
			s->u.string.at++;
		break;
	
	case StreamDynamic:
		if ((c = s->u.dyn.buf[s->u.dyn.at]))
			s->u.dyn.at++;
		break;
	
	default:
		ErCreate("StreamGetc: Bogus Stream!");
		return EOS;
	}

	if (c == '\n')
		s->lineno++;

	return c;
}

static inline
void StreamPutc(Stream s, int c)
{
	if (!s || !c)
		return;

	switch(s->type) {

	case StreamFile:
		putc(c, s->u.file.fp);
		break;

	case StreamString:
		*(s->u.string.at++) = c;
		*(s->u.string.at)   = '\0';
		break;
	
	case StreamDynamic:
		if (s->u.dyn.at >= s->u.dyn.maxLen)
			StreamDynamicGrow(s);
		s->u.dyn.buf[s->u.dyn.at++] = c;
		if (s->u.dyn.at > s->u.dyn.len) {
			s->u.dyn.len = s->u.dyn.at;
			s->u.dyn.buf[s->u.dyn.len] = '\0';
		}
		break;

	default:
		ErCreate("StreamPutc: Bogus Stream!");
		return;
	}

	if (c == '\n')
		s->lineno++;
}

static inline
void StreamUngetc(Stream s, int c)
{
	if (!s || !c)
		return;

	switch(s->type) {

	case StreamFile:
		ungetc(c, s->u.file.fp);
		break;

	case StreamString:
		s->u.string.at--;
		break;

	case StreamDynamic:
		s->u.dyn.at--;
		break;

	default:
		ErCreate("StreamUngetc: Bogus Stream!");
		return;
	}

	if (c == '\n')
		s->lineno--;
}

#endif

